/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2008. All Rights Reserved.
 * 
 * Note to U.S. Government Users Restricted Rights:  
 * Use, duplication or disclosure restricted by GSA ADP Schedule 
 * Contract with IBM Corp. 
 *******************************************************************************/
package junit.awtui;

import java.awt.*;
/**
 * A progress bar to show progress on
 */
public class ProgressBar extends Canvas {
	private static final long serialVersionUID = -9157418243100930959L;
	
	private boolean fError= false;
	private int fTotal= 0;
	private int fProgress= 0;
	private int fProgressX= 0;

	/**
	 * Make a new progress bar
	 * To specify the amount of work to be tracked use {@link #start(int)}
	 */
	public ProgressBar() {
		super();
		setSize(20, 30);
	}
	
    /**
     * Paints the progress bar
     * 
     * @param      g   the specified Graphics context
     */
	final public void paint(Graphics g) {
		paintBackground(g);
		paintStatus(g);
	}
	
	/**
	 * paint the background of the progress bar
	 */
	private void paintBackground(Graphics g) {
		g.setColor(SystemColor.control);
		Rectangle r= getBounds();
		g.fillRect(0, 0, r.width, r.height);
		g.setColor(Color.darkGray);
		g.drawLine(0, 0, r.width-1, 0);
		g.drawLine(0, 0, 0, r.height-1);
		g.setColor(Color.white);
		g.drawLine(r.width-1, 0, r.width-1, r.height-1);
		g.drawLine(0, r.height-1, r.width-1, r.height-1);
	}
	
	private void paintStatus(Graphics g) {
		g.setColor(fError ? Color.red : Color.green);
		Rectangle r= new Rectangle(0, 0, fProgressX, getBounds().height);
		g.fillRect(1, 1, r.width-1, r.height-2);
	}
	
	private void paintStep(int startX, int endX) {
		repaint(startX, 1, endX-startX, getBounds().height-2);
	}
	
	/**
	 * Resets the progress made in the progress bar back to 0 again.
	 */
	public void reset() {
		fProgressX= 1;
		fProgress= 0;
		fError= false;
		paint(getGraphics());
	}
	
	private int scale(int value) {
		// make it more obscure
		return fTotal > 0 ?	Math.max(1, value*(getBounds().width-1)/fTotal) : value;
	}
	
    /**
     * Moves and resizes this component. The new location of the top-left
     * corner is specified by <code>x</code> and <code>y</code>, and the
     * new size is specified by <code>width</code> and <code>height</code>.
     * @param x the new <i>x</i>-coordinate of this component
     * @param y the new <i>y</i>-coordinate of this component
     * @param width the new <code>width</code> of this component
     * @param height the new <code>height</code> of this 
     *          component
     */
	final public void setBounds(int x, int y, int w, int h) {
		super.setBounds(x, y, w, h);
		// adjust the progress shown made based upon the new bounds
		// If the size of the window changes, then we need to show
		// the same proportion of progress completed.
		fProgressX= scale(fProgress);
	}
	
	/**
	 * Resets the progress bar and changes the total amount of
	 * work to be done.
	 * @param total The total amount of progress to be made.
	 * Should be non-negative. 
	 */
	public void start(int total) {
		if (total < 0) {
			throw new IllegalArgumentException("Total amount of progress must not be negative"); //$NON-NLS-1$
		}
		reset();
		
		// for safety do it after we reset.
		fTotal= total;
	}
	
	/**
	 * Step forward in the progress.  Indicate if the test
	 * run was successful or not.
	 * @param successful Whether the test run was successful or not.
	 */
	public void step(boolean successful) {
		// don't overflow the progress bar if it goes over 100%
		if (fProgress < fTotal) {
			fProgress++;
		}
		int x= fProgressX;

		fProgressX= scale(fProgress);

		if (!fError && !successful) {
			fError= true;
			// we want to show the whole progress bar
			// as failed.
			x= 1;
		}
		paintStep(x, fProgressX);
	}
}